package sample_project;



import java.awt.*;
import java.util.List;

import javax.swing.*;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.dial.*;
import org.jfree.data.general.DefaultValueDataset;
import org.jfree.data.general.ValueDataset;

/**
 * A simple example on how to use a dial in a dynamic setting.
 * @author lajthabalazs
 *
 */
public class Dial extends JFrame
{
	private static final long serialVersionUID = 432660291577382288L;

	static class DialPanel extends JPanel {

		private static final long serialVersionUID = -3920165641606935329L;
		JSlider slider;
		DefaultValueDataset dataset;

		/**
		 * Creates a minimalist dial. There are many fancy options including background fills, threshold levels, multiple and/or styled pins...
		 * @param title The title of the chart
		 * @param label Label indicated on the dial
		 * @param valuedataset Dataset containing data for each - in this case the single - pin
		 * @param min Minimum value
		 * @param max Maximum value
		 * @param step Difference between marked steps 
		 * @param minorTickCount Number of small ticks between two market steps
		 * @return
		 */
		public static JFreeChart createStandardDialChart(String title, String label, ValueDataset valuedataset, double min, double max, double step, int minorTickCount)
		{
			DialPlot dialplot = new DialPlot();
			dialplot.setDataset(valuedataset);
			dialplot.setDialFrame(new StandardDialFrame());
			dialplot.setBackground(new DialBackground());
			DialTextAnnotation dialtextannotation = new DialTextAnnotation(label);
			dialtextannotation.setFont(new Font("Dialog", 1, 14));
			dialtextannotation.setRadius(0.69999999999999996D);
			dialplot.addLayer(dialtextannotation);
			DialValueIndicator dialvalueindicator = new DialValueIndicator(0);
			dialplot.addLayer(dialvalueindicator);
			StandardDialScale standarddialscale = new StandardDialScale(min, max, -120D, -300D, 10D, 4);
			standarddialscale.setMajorTickIncrement(step);
			standarddialscale.setMinorTickCount(minorTickCount);
			standarddialscale.setTickRadius(0.88D);
			standarddialscale.setTickLabelOffset(0.14999999999999999D);
			standarddialscale.setTickLabelFont(new Font("Dialog", 0, 14));
			dialplot.addScale(0, standarddialscale);
			dialplot.addPointer(new org.jfree.chart.plot.dial.DialPointer.Pin());
			DialCap dialcap = new DialCap();
			dialplot.setCap(dialcap);
			return new JFreeChart(title, dialplot);
		}

		public DialPanel()
		{
			super(new BorderLayout());
			dataset = new DefaultValueDataset(0D);
			JFreeChart jfreechart = createStandardDialChart("Swing Worker", "Aggregated publishes", dataset, 0D, 100D, 10D, 4);
			ChartPanel chartpanel = new ChartPanel(jfreechart);
			chartpanel.setPreferredSize(new Dimension(400, 400));
			add(chartpanel);
		}
		
		public void setValue(double value) {
			dataset.setValue(value);
		}
	}

	/**
	 * Swing worker is the way to go when dealing with background tasks in Java Swing.
	 * This class publishes random data to the event thread in each millisecond
	 * @author lajthabalazs
	 *
	 */
	private class BackgroundTask extends SwingWorker<Integer, Double> {
		private boolean running = true;
		private DialPanel panel;
		
		public BackgroundTask(DialPanel panel) {
			this.panel = panel;
		}

		@Override
		protected Integer doInBackground() throws Exception {
			while(running) {
				try {
					Thread.sleep(1);
					publish(Math.random() * 100);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			return 1;
		}
		
		@Override
		protected void process(List<Double> chunks) {
			// Displays the number of chunks aggregated for this process call
			panel.setValue(chunks.size());
		}
	}
	
	/**
	 * Creates a frame with a single dial
	 * @param title Title of the frame
	 */
	public Dial(String title)
	{
		super(title);
		setDefaultCloseOperation(3);
		DialPanel panel = createDemoPanel();
		setContentPane(panel);
		new BackgroundTask(panel).execute();
	}

	public static DialPanel createDemoPanel()
	{
		return new DialPanel();
	}

	public static void main(String args[])
	{
		Dial dialdemo1 = new Dial("JFreeChart - Number of aggregated swing publishes");
		dialdemo1.pack();
		dialdemo1.setVisible(true);
	}
}